<?php

declare(strict_types=1);

namespace App\Modules;

use App\Bot\TelegramBot;

final class RatingModule extends BaseModule
{
    public function supports(string $command): bool
    {
        return false;
    }

    public function handle(string $command, array $update, TelegramBot $bot): void
    {
        // This module is callback-only.
    }

    public function handleCallback(string $data, array $update, TelegramBot $bot): void
    {
        if (strpos($data, 'rate:') !== 0) {
            return;
        }

        $parts = explode(':', $data);
        if (count($parts) < 5) {
            return;
        }

        [, $chatIdRaw, $topicIdRaw, $messageIdRaw, $starsRaw] = $parts;

        $chatId = (int) $chatIdRaw;
        $topicId = (int) $topicIdRaw;
        $messageId = (int) $messageIdRaw;
        $stars = (int) $starsRaw;

        if ($chatId === 0 || $messageId === 0 || $stars < 1 || $stars > 5) {
            return;
        }

        $callback = $update['callback_query'] ?? [];
        $from = $callback['from'] ?? [];
        $message = $callback['message'] ?? [];
        $replyTo = $message['reply_to_message'] ?? [];

        $callbackId = isset($callback['id']) ? (string) $callback['id'] : '';

        if (!isset($from['id'])) {
            return;
        }

        $voterId = (int) $from['id'];
        $authorId = isset($replyTo['from']['id']) ? (int) $replyTo['from']['id'] : 0;

        if ($authorId !== 0 && $authorId === $voterId) {
            if ($callbackId !== '') {
                $bot->getClient()->request('answerCallbackQuery', [
                    'callback_query_id' => $callbackId,
                    'text' => 'نمی‌توانی به تحلیل خودت امتیاز بدهی.',
                    'show_alert' => false,
                ]);
            }
            return;
        }

        $storage = $bot->getStorage();

        $didVote = false;
        $alreadyVoted = false;

        $storage->update('ratings', function (array $ratings) use ($chatId, $topicId, $messageId, $stars, $voterId, $replyTo, &$didVote, &$alreadyVoted): array {
            if (!isset($ratings[$chatId])) {
                $ratings[$chatId] = [];
            }

            if (!isset($ratings[$chatId][$messageId])) {
                $now = (new \DateTimeImmutable())->format('Y-m-d H:i:s');
                $author = $replyTo['from'] ?? [];

                $ratings[$chatId][$messageId] = [
                    'chat_id' => $chatId,
                    'topic_id' => $topicId,
                    'message_id' => $messageId,
                    'author_id' => isset($author['id']) ? (int) $author['id'] : 0,
                    'author_username' => (string) ($author['username'] ?? ''),
                    'author_first_name' => (string) ($author['first_name'] ?? ''),
                    'created_at' => $now,
                    'updated_at' => $now,
                    'votes' => [],
                ];
            }

            $entry =& $ratings[$chatId][$messageId];

            if (!isset($entry['votes']) || !is_array($entry['votes'])) {
                $entry['votes'] = [];
            }

            if (isset($entry['votes'][$voterId])) {
                $alreadyVoted = true;
                return $ratings;
            }

            // محاسبه سطح فعلی رأی‌دهنده برای وزن‌دهی رأی
            $totalScore = 0;
            if (isset($ratings[$chatId]) && is_array($ratings[$chatId])) {
                foreach ($ratings[$chatId] as $r) {
                    if (!is_array($r)) {
                        continue;
                    }
                    if ((int) ($r['author_id'] ?? 0) !== $voterId) {
                        continue;
                    }
                    $st = $r['stats'] ?? [];
                    $totalScore += (int) ($st['sum'] ?? 0);
                }
            }

            $weight = 1.0;
            if ($totalScore >= 1000) {
                $weight = 2.0;
            } elseif ($totalScore >= 600) {
                $weight = 1.5;
            } elseif ($totalScore >= 300) {
                $weight = 1.2;
            } else {
                $weight = 1.0;
            }

            $now = (new \DateTimeImmutable())->format('Y-m-d H:i:s');

            $entry['votes'][$voterId] = [
                'user_id' => $voterId,
                'stars' => $stars,
                'voted_at' => $now,
                'weight' => $weight,
            ];

            $didVote = true;

            $sum = 0;
            $count = 0;
            $weightedSum = 0.0;
            $weightedCount = 0.0;

            foreach ($entry['votes'] as $vote) {
                $s = (int) ($vote['stars'] ?? 0);
                $w = isset($vote['weight']) ? (float) $vote['weight'] : 1.0;

                $sum += $s;
                $count++;

                $weightedSum += $s * $w;
                $weightedCount += $w;
            }

            $avg = $count > 0 ? round($sum / $count, 2) : 0.0;
            $avgWeighted = $weightedCount > 0 ? round($weightedSum / $weightedCount, 2) : 0.0;

            $entry['stats'] = [
                'count' => $count,
                'sum' => $sum,
                'avg' => $avg,
                'weighted_sum' => $weightedSum,
                'weighted_count' => $weightedCount,
                'avg_weighted' => $avgWeighted,
            ];

            $entry['updated_at'] = $now;

            return $ratings;
        });

        if ($didVote) {
            $settings = $storage->load('settings');
            if (isset($settings['main_group']['id']) && (int) $settings['main_group']['id'] === $chatId) {
                $now = (new \DateTimeImmutable())->format('Y-m-d H:i:s');

                $storage->update('inactivity_status', function (array $data) use ($chatId, $voterId, $now): array {
                    if (!isset($data[$chatId])) {
                        $data[$chatId] = [];
                    }

                    if (!isset($data[$chatId][$voterId]) || !is_array($data[$chatId][$voterId])) {
                        $data[$chatId][$voterId] = [
                            'user_id' => $voterId,
                            'chat_id' => $chatId,
                            'last_activity_at' => $now,
                            'inactivity_days' => 0,
                            'pv_available' => null,
                            'warned_stage' => 0,
                            'last_warning_at' => '',
                            'admin_notified' => false,
                        ];

                        return $data;
                    }

                    $entry =& $data[$chatId][$voterId];
                    $entry['last_activity_at'] = $now;
                    $entry['inactivity_days'] = 0;
                    $entry['warned_stage'] = 0;
                    $entry['admin_notified'] = false;

                    return $data;
                });
            }
        }

        if ($callbackId !== '') {
            $text = '';
            if ($alreadyVoted) {
                $text = 'شما قبلاً به این تحلیل رأی داده‌اید.';
            } elseif ($didVote) {
                $text = 'امتیاز شما ثبت شد: ' . $stars . ' ⭐';
            }

            if ($text !== '') {
                $bot->getClient()->request('answerCallbackQuery', [
                    'callback_query_id' => $callbackId,
                    'text' => $text,
                    'show_alert' => false,
                ]);
            }
        }
    }
}
