<?php

declare(strict_types=1);

namespace App\Modules;

use App\Bot\TelegramBot;

final class InactivityModule extends BaseModule
{
    public function supports(string $command): bool
    {
        return false;
    }

    public function handle(string $command, array $update, TelegramBot $bot): void
    {
    }

    public function handleCallback(string $data, array $update, TelegramBot $bot): void
    {
        if (strpos($data, 'inactivity:') !== 0) {
            return;
        }

        $action = substr($data, strlen('inactivity:'));
        if ($action === 'summary') {
            $this->handleSummary($update, $bot);
        }
    }

    private function handleSummary(array $update, TelegramBot $bot): void
    {
        $callback = $update['callback_query'] ?? [];
        $from = $callback['from'] ?? [];
        $message = $callback['message'] ?? [];
        $chat = $message['chat'] ?? [];

        if (!isset($from['id'], $chat['id'])) {
            return;
        }

        $config = $bot->getConfig();
        $adminIds = $config['admin_ids'] ?? [];
        $userId = (int) $from['id'];
        $isAdmin = in_array($userId, array_map('intval', $adminIds), true);
        if (!$isAdmin) {
            return;
        }

        $chatId = (int) $chat['id'];
        $storage = $bot->getStorage();

        $settings = $storage->load('settings');
        if (!isset($settings['main_group']['id'])) {
            $bot->getClient()->sendMessage($chatId, 'ابتدا باید یک گروه اصلی ثبت شود. در گروه ترید بنویس: "نصب ربات".');
            return;
        }

        $mainGroupId = (int) $settings['main_group']['id'];

        $result = $this->runCheck($bot, $mainGroupId, $chatId);

        $stage1 = $result['stage1'] ?? [];
        $stage2 = $result['stage2'] ?? [];
        $stage3 = $result['stage3'] ?? [];

        $lines = [];
        $lines[] = '<b>🚨 وضعیت عدم فعالیت هنرجویان</b>';
        $lines[] = '';
        $lines[] = 'مرحله ۱ (هشدار به هنرجو): ' . count($stage1) . ' نفر';
        $lines[] = 'مرحله ۲ (گزارش به کوچ): ' . count($stage2) . ' نفر';
        $lines[] = 'مرحله ۳ (در معرض رها کردن): ' . count($stage3) . ' نفر';

        $formatUserLine = function (array $row): string {
            $name = (string) ($row['name'] ?? '');
            $id = (int) ($row['user_id'] ?? 0);
            $days = (int) ($row['days'] ?? 0);
            return '• ' . $name . ' (شناسه: <code>' . $id . '</code>) - ' . $days . ' روز غیرفعال';
        };

        if (!empty($stage3)) {
            $lines[] = '';
            $lines[] = '🚨 مرحله ۳:';
            foreach ($stage3 as $row) {
                $lines[] = $formatUserLine($row);
            }
        }

        if (!empty($stage2)) {
            $lines[] = '';
            $lines[] = '⚠️ مرحله ۲:';
            foreach ($stage2 as $row) {
                $lines[] = $formatUserLine($row);
            }
        }

        if (!empty($stage1)) {
            $lines[] = '';
            $lines[] = '⏰ مرحله ۱:';
            foreach ($stage1 as $row) {
                $lines[] = $formatUserLine($row);
            }
        }

        $bot->getClient()->sendMessage($chatId, implode("\n", $lines));
    }

    private function runCheck(TelegramBot $bot, int $mainGroupId, int $adminChatId): array
    {
        $storage = $bot->getStorage();
        $status = $storage->load('inactivity_status');
        $users = $storage->load('users');
        $config = $bot->getConfig();
        $adminIds = $config['admin_ids'] ?? [];
        if (!is_array($adminIds)) {
            $adminIds = [];
        }
        $adminIds = array_map('intval', $adminIds);

        if (!is_array($status)) {
            $status = [];
        }

        if (!isset($status[$mainGroupId]) || !is_array($status[$mainGroupId])) {
            return [
                'stage1' => [],
                'stage2' => [],
                'stage3' => [],
            ];
        }

        $now = new \DateTimeImmutable('now');
        $nowStr = $now->format('Y-m-d H:i:s');

        $stage1 = [];
        $stage2 = [];
        $stage3 = [];

        foreach ($status[$mainGroupId] as $userId => &$entry) {
            if (!is_array($entry)) {
                continue;
            }

            $userId = (int) $userId;

            if (in_array($userId, $adminIds, true)) {
                continue;
            }

            $last = (string) ($entry['last_activity_at'] ?? '');
            if ($last === '') {
                continue;
            }

            try {
                $dt = new \DateTimeImmutable($last);
                $diff = $now->diff($dt);
                $days = (int) $diff->days;
            } catch (\Throwable $e) {
                continue;
            }

            $entry['inactivity_days'] = $days;

            $lastWarningAt = (string) ($entry['last_warning_at'] ?? '');
            $canWarn = true;
            if ($lastWarningAt !== '') {
                try {
                    $lw = new \DateTimeImmutable($lastWarningAt);
                    $wdiff = $now->getTimestamp() - $lw->getTimestamp();
                    if ($wdiff < 86400) {
                        $canWarn = false;
                    }
                } catch (\Throwable $e) {
                    $canWarn = true;
                }
            }

            $name = '';
            if (isset($users[$userId]) && is_array($users[$userId])) {
                $u = $users[$userId];
                if (isset($u['first_name']) && $u['first_name'] !== '') {
                    $name = (string) $u['first_name'];
                } elseif (isset($u['username']) && $u['username'] !== '') {
                    $name = '@' . (string) $u['username'];
                }
            }
            if ($name === '') {
                $name = 'کاربر ' . $userId;
            }

            if ($days >= 14) {
                if ((int) ($entry['warned_stage'] ?? 0) < 3 && $canWarn) {
                    $this->notifyStage3($bot, $adminChatId, $userId, $name, $days);
                    $entry['warned_stage'] = 3;
                    $entry['last_warning_at'] = $nowStr;
                    $entry['admin_notified'] = true;
                }
                $stage3[] = [
                    'user_id' => $userId,
                    'name' => $name,
                    'days' => $days,
                ];
                continue;
            }

            if ($days >= 10) {
                if ((int) ($entry['warned_stage'] ?? 0) < 2 && $canWarn) {
                    $this->notifyStage2($bot, $adminChatId, $userId, $name, $days);
                    $entry['warned_stage'] = 2;
                    $entry['last_warning_at'] = $nowStr;
                    $entry['admin_notified'] = true;
                }
                $stage2[] = [
                    'user_id' => $userId,
                    'name' => $name,
                    'days' => $days,
                ];
                continue;
            }

            if ($days >= 7) {
                if ((int) ($entry['warned_stage'] ?? 0) < 1 && $canWarn) {
                    $this->notifyStage1($bot, $mainGroupId, $userId, $name, $entry, $nowStr);
                }
                $stage1[] = [
                    'user_id' => $userId,
                    'name' => $name,
                    'days' => $days,
                ];
            }
        }
        unset($entry);

        $status[$mainGroupId] = $status[$mainGroupId];
        $storage->save('inactivity_status', $status);

        return [
            'stage1' => $stage1,
            'stage2' => $stage2,
            'stage3' => $stage3,
        ];
    }

    private function notifyStage1(TelegramBot $bot, int $mainGroupId, int $userId, string $name, array &$entry, string $nowStr): void
    {
        $client = $bot->getClient();

        $pvAvailable = $entry['pv_available'] ?? null;
        $sent = false;

        if ($pvAvailable === null || $pvAvailable === true) {
            try {
                $text = '👋 ' . $name . " عزیز\n" .
                    'مدتی است در گروه فعالیتی از تو ثبت نشده. اگر درگیر هستی یا سوالی داری، حتماً یک پیام کوتاه بفرست تا بدانیم همراه ما هستی.';
                $client->sendMessage($userId, $text);
                $entry['pv_available'] = true;
                $sent = true;
            } catch (\Throwable $e) {
                $entry['pv_available'] = false;
            }
        }

        if (!$sent) {
            $link = '<a href="tg://user?id=' . $userId . '">' . $name . '</a>';
            $text = '👋 ' . $link . " عزیز\n" .
                'مدتی است در گروه فعالیتی از تو ثبت نشده. اگر درگیر هستی یا سوالی داری، یک پیام در گروه یا پی‌وی بفرست تا در جریان باشیم.';
            $client->sendMessage($mainGroupId, $text);
        }

        $entry['warned_stage'] = 1;
        $entry['last_warning_at'] = $nowStr;
        $entry['admin_notified'] = false;
    }

    private function notifyStage2(TelegramBot $bot, int $adminChatId, int $userId, string $name, int $days): void
    {
        $text = '⚠️ هنرجو در آستانه رها کردن است:' . "\n" .
            '👤 ' . $name . ' (شناسه: <code>' . $userId . '</code>)' . "\n" .
            '⏱ مدت عدم فعالیت: حدود ' . $days . ' روز.' . "\n" .
            'لطفاً وضعیت او را بررسی کن و در صورت نیاز با او تماس بگیر. تصمیم نهایی کاملاً با کوچ است.';

        $bot->getClient()->sendMessage($adminChatId, $text);
    }

    private function notifyStage3(TelegramBot $bot, int $adminChatId, int $userId, string $name, int $days): void
    {
        $text = '🚨 یادآوری مهم درباره هنرجوی غیرفعال:' . "\n" .
            '👤 ' . $name . ' (شناسه: <code>' . $userId . '</code>)' . "\n" .
            '⏱ مدت عدم فعالیت: بیش از ' . $days . ' روز.' . "\n" .
            'در صورت صلاحدید می‌توانی درباره ادامه حضور او تصمیم بگیری. ربات هیچ اقدام خودکاری انجام نمی‌دهد.';

        $bot->getClient()->sendMessage($adminChatId, $text);
    }
}
