<?php

declare(strict_types=1);

namespace App\Modules;

use App\Bot\TelegramBot;

final class CoachDashboardModule extends BaseModule
{
    public function supports(string $command): bool
    {
        return false;
    }

    public function handle(string $command, array $update, TelegramBot $bot): void
    {
    }

    public function handleCallback(string $data, array $update, TelegramBot $bot): void
    {
        if (strpos($data, 'coach:') !== 0) {
            return;
        }

        $action = substr($data, strlen('coach:'));
        if ($action === 'summary') {
            $this->handleSummary($update, $bot);
        }
    }

    private function handleSummary(array $update, TelegramBot $bot): void
    {
        $callback = $update['callback_query'] ?? [];
        $from = $callback['from'] ?? [];
        $message = $callback['message'] ?? [];
        $chat = $message['chat'] ?? [];

        if (!isset($from['id'], $chat['id'])) {
            return;
        }

        $config = $bot->getConfig();
        $adminIds = $config['admin_ids'] ?? [];
        $userId = (int) $from['id'];
        $isAdmin = in_array($userId, array_map('intval', $adminIds), true);
        if (!$isAdmin) {
            return;
        }

        $chatId = (int) $chat['id'];
        $storage = $bot->getStorage();

        $settings = $storage->load('settings');
        if (!isset($settings['main_group']['id'])) {
            $bot->getClient()->sendMessage($chatId, 'ابتدا باید یک گروه اصلی ثبت شود. در گروه ترید بنویس: "نصب ربات".');
            return;
        }

        $mainGroupId = (int) $settings['main_group']['id'];

        $summary = $this->buildSummary($bot, $mainGroupId);

        $stable = $summary['stable'] ?? [];
        $needs = $summary['needs'] ?? [];
        $risk = $summary['risk'] ?? [];

        $lines = [];
        $lines[] = '<b>🎯 اولویت توجه کوچ</b>';

        $format = function (array $row): string {
            $name = (string) ($row['name'] ?? '');
            $id = (int) ($row['user_id'] ?? 0);
            $behavior = (string) ($row['behavior_label_text'] ?? '');
            $trend = (string) ($row['trend_label'] ?? '');
            $inact = (int) ($row['inactivity_days'] ?? 0);
            return '• ' . $name . ' (شناسه: <code>' . $id . '</code>) - ' . $behavior . ' | ' . $trend . ' | عدم فعالیت حدود ' . $inact . ' روز';
        };

        if (!empty($risk)) {
            $lines[] = '';
            $lines[] = '🚨 در معرض رها کردن:';
            foreach ($risk as $row) {
                $lines[] = $format($row);
            }
        }

        if (!empty($needs)) {
            $lines[] = '';
            $lines[] = '⚠️ نیازمند توجه کوچ:';
            foreach ($needs as $row) {
                $lines[] = $format($row);
            }
        }

        if (!empty($stable)) {
            $lines[] = '';
            $lines[] = '✅ رشد پایدار و حضور خوب:';
            foreach ($stable as $row) {
                $lines[] = $format($row);
            }
        }

        if (count($lines) === 1) {
            $lines[] = '';
            $lines[] = 'هنوز داده کافی برای ساخت داشبورد کوچ وجود ندارد. بعد از چند هفته فعالیت، این بخش کامل‌تر می‌شود.';
        }

        $bot->getClient()->sendMessage($chatId, implode("\n", $lines));
    }

    private function buildSummary(TelegramBot $bot, int $mainGroupId): array
    {
        $storage = $bot->getStorage();
        $users = $storage->load('users');
        $behaviorProfiles = $storage->load('behavior_profiles');
        $progressHistory = $storage->load('progress_history');
        $inactivityStatus = $storage->load('inactivity_status');
        $config = $bot->getConfig();
        $adminIds = $config['admin_ids'] ?? [];
        if (!is_array($adminIds)) {
            $adminIds = [];
        }
        $adminIds = array_map('intval', $adminIds);

        if (!is_array($users)) {
            $users = [];
        }
        if (!is_array($behaviorProfiles)) {
            $behaviorProfiles = [];
        }
        if (!is_array($progressHistory)) {
            $progressHistory = [];
        }
        if (!is_array($inactivityStatus)) {
            $inactivityStatus = [];
        }

        $now = new \DateTimeImmutable('now');

        $labelMap = [
            'disciplined' => 'منظم ✅',
            'consistent' => 'پایدار 📈',
            'passive' => 'کم‌فعال 💤',
            'overactive' => 'پرپیام 💬',
            'emotional' => 'هیجانی 🎢',
        ];

        $allUserIds = [];
        if (isset($behaviorProfiles[$mainGroupId]) && is_array($behaviorProfiles[$mainGroupId])) {
            foreach ($behaviorProfiles[$mainGroupId] as $uid => $_) {
                $allUserIds[(int) $uid] = true;
            }
        }
        if (isset($progressHistory[$mainGroupId]) && is_array($progressHistory[$mainGroupId])) {
            foreach ($progressHistory[$mainGroupId] as $uid => $_) {
                $allUserIds[(int) $uid] = true;
            }
        }
        if (isset($inactivityStatus[$mainGroupId]) && is_array($inactivityStatus[$mainGroupId])) {
            foreach ($inactivityStatus[$mainGroupId] as $uid => $_) {
                $allUserIds[(int) $uid] = true;
            }
        }

        $stable = [];
        $needs = [];
        $risk = [];

        foreach (array_keys($allUserIds) as $uid) {
            $uid = (int) $uid;

            if (in_array($uid, $adminIds, true)) {
                continue;
            }

            $name = '';
            if (isset($users[$uid]) && is_array($users[$uid])) {
                $u = $users[$uid];
                if (!empty($u['first_name'])) {
                    $name = (string) $u['first_name'];
                } elseif (!empty($u['username'])) {
                    $name = '@' . (string) $u['username'];
                }
            }
            if ($name === '') {
                $name = 'کاربر ' . $uid;
            }

            $behaviorLabel = '';
            $behaviorText = '';
            if (isset($behaviorProfiles[$mainGroupId][$uid]) && is_array($behaviorProfiles[$mainGroupId][$uid])) {
                $bp = $behaviorProfiles[$mainGroupId][$uid];
                $behaviorLabel = (string) ($bp['behavior_label'] ?? '');
                if (isset($labelMap[$behaviorLabel])) {
                    $behaviorText = $labelMap[$behaviorLabel];
                } else {
                    $behaviorText = $behaviorLabel;
                }
            }

            $inactivityDays = 0;
            $warnedStage = 0;
            if (isset($inactivityStatus[$mainGroupId][$uid]) && is_array($inactivityStatus[$mainGroupId][$uid])) {
                $st = $inactivityStatus[$mainGroupId][$uid];
                $warnedStage = (int) ($st['warned_stage'] ?? 0);
                $last = (string) ($st['last_activity_at'] ?? '');
                if ($last !== '') {
                    try {
                        $dt = new \DateTimeImmutable($last);
                        $diff = $now->diff($dt);
                        $inactivityDays = (int) $diff->days;
                    } catch (\Throwable $e) {
                        $inactivityDays = (int) ($st['inactivity_days'] ?? 0);
                    }
                } else {
                    $inactivityDays = (int) ($st['inactivity_days'] ?? 0);
                }
            }

            $lastScore = 0;
            $prevScore = 0;
            if (isset($progressHistory[$mainGroupId][$uid]) && is_array($progressHistory[$mainGroupId][$uid]) && !empty($progressHistory[$mainGroupId][$uid])) {
                $list = $progressHistory[$mainGroupId][$uid];
                $count = count($list);
                $last = $list[$count - 1];
                if (is_array($last)) {
                    $lastScore = (int) ($last['activity_score'] ?? 0);
                }
                if ($count >= 2) {
                    $prev = $list[$count - 2];
                    if (is_array($prev)) {
                        $prevScore = (int) ($prev['activity_score'] ?? 0);
                    }
                } else {
                    $prevScore = $lastScore;
                }
            }

            $diff = $lastScore - $prevScore;
            $trendLabel = 'روند نامشخص';
            if ($diff > 5) {
                $trendLabel = 'روند صعودی 📈';
            } elseif ($diff < -5) {
                $trendLabel = 'روند نزولی 📉';
            } elseif ($lastScore > 0) {
                $trendLabel = 'روند تقریباً ثابت ➖';
            }

            $row = [
                'user_id' => $uid,
                'name' => $name,
                'behavior_label' => $behaviorLabel,
                'behavior_label_text' => $behaviorText,
                'inactivity_days' => $inactivityDays,
                'warned_stage' => $warnedStage,
                'last_score' => $lastScore,
                'prev_score' => $prevScore,
                'trend_label' => $trendLabel,
            ];

            $isRisk = false;
            $isNeeds = false;
            $isStable = false;

            if ($inactivityDays >= 14 || $warnedStage >= 3) {
                $isRisk = true;
            } elseif ($behaviorLabel === 'passive' && $inactivityDays >= 10) {
                $isRisk = true;
            }

            if (!$isRisk) {
                if ($diff < -5 || $behaviorLabel === 'emotional') {
                    $isNeeds = true;
                } elseif ($behaviorLabel === 'overactive' && $lastScore < 30) {
                    $isNeeds = true;
                }
            }

            if (!$isRisk && !$isNeeds) {
                if (in_array($behaviorLabel, ['disciplined', 'consistent'], true) && $inactivityDays < 7 && $lastScore >= 20) {
                    $isStable = true;
                }
            }

            if ($isRisk) {
                $risk[] = $row;
            } elseif ($isNeeds) {
                $needs[] = $row;
            } elseif ($isStable) {
                $stable[] = $row;
            }
        }

        return [
            'stable' => $stable,
            'needs' => $needs,
            'risk' => $risk,
        ];
    }
}
