<?php

declare(strict_types=1);

namespace App\Bot;

use App\Core\JsonStorage;
use App\Modules\ModuleInterface;
use App\Modules\HelpModule;
use App\Modules\AdminModule;
use App\Modules\RatingModule;
use App\Modules\InactivityModule;
use App\Modules\BehaviorAnalysisModule;
use App\Modules\CoachDashboardModule;
use App\Modules\LeaderboardModule;

final class TelegramBot
{
    private array $config;
    private JsonStorage $storage;
    private TelegramClient $client;
    private array $modules = [];

    public function __construct(array $config, JsonStorage $storage)
    {
        $this->config = $config;
        $this->storage = $storage;
        $this->client = new TelegramClient($config['bot_token']);
        $this->registerDefaultModules();
    }

    public function getClient(): TelegramClient
    {
        return $this->client;
    }

    public function getStorage(): JsonStorage
    {
        return $this->storage;
    }

    public function getConfig(): array
    {
        return $this->config;
    }

    private function registerDefaultModules(): void
    {
        $this->modules[] = new HelpModule($this);
        $this->modules[] = new AdminModule($this);
        $this->modules[] = new RatingModule($this);
        $this->modules[] = new InactivityModule($this);
        $this->modules[] = new BehaviorAnalysisModule($this);
        $this->modules[] = new CoachDashboardModule($this);
        $this->modules[] = new LeaderboardModule($this);
    }

    public function addModule(ModuleInterface $module): void
    {
        $this->modules[] = $module;
    }

    public function handleUpdate(array $update): void
    {
        try {
            if (isset($update['callback_query'])) {
                $this->handleCallbackUpdate($update);
                return;
            }
            if (isset($update['message'])) {
                $this->handleMessageUpdate($update);
            }
        } catch (\Throwable $e) {
            $this->logError($e->getMessage());
        }
    }

    private function handleMessageUpdate(array $update): void
    {
        $message = $update['message'];
        if (isset($message['from']) && is_array($message['from'])) {
            $this->registerUser($message['from']);
        }
        $this->logTopicActivity($message);
        $this->attachTradingRatingPanel($message);
        if (!isset($message['text'])) {
            return;
        }
        $text = trim((string) $message['text']);
        if ($text === '') {
            return;
        }

        if ($text[0] === '/') {
            $this->dispatchCommand($text, $update);
            return;
        }

        $this->handleAdminKeywords($text, $update);

        $this->handleUserKeywords($text, $update);
    }

    private function handleCallbackUpdate(array $update): void
    {
        $callback = $update['callback_query'];
        $data = isset($callback['data']) ? (string) $callback['data'] : '';
        if ($data === '') {
            return;
        }

        foreach ($this->modules as $module) {
            if (method_exists($module, 'handleCallback')) {
                $module->handleCallback($data, $update, $this);
            }
        }
    }

    private function logTopicActivity(array $message): void
    {
        if (!isset($message['chat']['id'], $message['from']['id'])) {
            return;
        }

        $chat = $message['chat'];
        $chatType = (string) ($chat['type'] ?? '');
        if ($chatType !== 'group' && $chatType !== 'supergroup') {
            return;
        }

        $chatId = (int) $chat['id'];
        $topicId = isset($message['message_thread_id']) ? (int) $message['message_thread_id'] : 0;
        $from = $message['from'];
        $userId = (int) $from['id'];
        $now = (new \DateTimeImmutable())->format('Y-m-d H:i:s');

        $text = isset($message['text']) ? (string) $message['text'] : '';
        $caption = isset($message['caption']) ? (string) $message['caption'] : '';

        $hasMedia = false;
        if ((isset($message['photo']) && is_array($message['photo']) && !empty($message['photo']))
            || isset($message['video'])
            || isset($message['animation'])
            || isset($message['document'])) {
            $hasMedia = true;
        }

        $hasLink = false;
        if (isset($message['entities']) && is_array($message['entities'])) {
            foreach ($message['entities'] as $entity) {
                if (!is_array($entity)) {
                    continue;
                }
                $type = (string) ($entity['type'] ?? '');
                if ($type === 'url' || $type === 'text_link') {
                    $hasLink = true;
                    break;
                }
            }
        }
        if (!$hasLink && isset($message['caption_entities']) && is_array($message['caption_entities'])) {
            foreach ($message['caption_entities'] as $entity) {
                if (!is_array($entity)) {
                    continue;
                }
                $type = (string) ($entity['type'] ?? '');
                if ($type === 'url' || $type === 'text_link') {
                    $hasLink = true;
                    break;
                }
            }
        }
        if (!$hasLink) {
            if (strpos($text, 'http://') !== false
                || strpos($text, 'https://') !== false
                || strpos($caption, 'http://') !== false
                || strpos($caption, 'https://') !== false) {
                $hasLink = true;
            }
        }

        $this->storage->update('topic_activity', function (array $data) use ($chatId, $topicId, $userId, $from, $now, $hasMedia, $hasLink): array {
            if (!isset($data[$chatId])) {
                $data[$chatId] = [];
            }

            if (!isset($data[$chatId][$topicId])) {
                $data[$chatId][$topicId] = [
                    'chat_id' => $chatId,
                    'topic_id' => $topicId,
                    'messages_total' => 0,
                    'media_total' => 0,
                    'links_total' => 0,
                    'first_message_at' => $now,
                    'last_message_at' => $now,
                    'users' => [],
                ];
            }

            $topic =& $data[$chatId][$topicId];
            if (!isset($topic['media_total'])) {
                $topic['media_total'] = 0;
            }
            if (!isset($topic['links_total'])) {
                $topic['links_total'] = 0;
            }
            $topic['messages_total']++;
            if (!isset($topic['first_message_at']) || $topic['first_message_at'] === '') {
                $topic['first_message_at'] = $now;
            }
            $topic['last_message_at'] = $now;

            if (!isset($topic['users'][$userId])) {
                $topic['users'][$userId] = [
                    'id' => $userId,
                    'message_count' => 0,
                    'first_message_at' => $now,
                    'last_message_at' => $now,
                    'daily_messages' => [],
                    'media_count' => 0,
                    'link_count' => 0,
                    'is_bot' => (bool) ($from['is_bot'] ?? false),
                    'first_name' => (string) ($from['first_name'] ?? ''),
                    'last_name' => (string) ($from['last_name'] ?? ''),
                    'username' => (string) ($from['username'] ?? ''),
                ];
            }

            $user =& $topic['users'][$userId];
            if (!isset($user['media_count'])) {
                $user['media_count'] = 0;
            }
            if (!isset($user['link_count'])) {
                $user['link_count'] = 0;
            }
            $user['message_count']++;
            if (!isset($user['first_message_at']) || $user['first_message_at'] === '') {
                $user['first_message_at'] = $now;
            }
            $user['last_message_at'] = $now;
            if (!isset($user['daily_messages']) || !is_array($user['daily_messages'])) {
                $user['daily_messages'] = [];
            }
            $dayKey = (new \DateTimeImmutable($now))->format('Y-m-d');
            if (!isset($user['daily_messages'][$dayKey])) {
                $user['daily_messages'][$dayKey] = 0;
            }
            $user['daily_messages'][$dayKey]++;
            $user['is_bot'] = (bool) ($from['is_bot'] ?? false);
            $user['first_name'] = (string) ($from['first_name'] ?? '');
            $user['last_name'] = (string) ($from['last_name'] ?? '');
            $user['username'] = (string) ($from['username'] ?? '');

            if ($hasMedia) {
                $topic['media_total']++;
                $user['media_count']++;
            }

            if ($hasLink) {
                $topic['links_total']++;
                $user['link_count']++;
            }

            return $data;
        });

        $this->updateInactivityOnActivity($chatId, $userId);
    }

    private function attachTradingRatingPanel(array $message): void
    {
        if (!isset($message['chat']['id'], $message['message_id'])) {
            return;
        }

        if (!isset($message['message_thread_id'])) {
            return;
        }

        $chat = $message['chat'];
        $chatType = (string) ($chat['type'] ?? '');
        if ($chatType !== 'group' && $chatType !== 'supergroup') {
            return;
        }

        $chatId = (int) $chat['id'];
        $topicId = (int) $message['message_thread_id'];
        $messageId = (int) $message['message_id'];

        $roles = $this->storage->load('topic_roles');
        if (!is_array($roles) || !isset($roles[$chatId]['trading']['topic_id'])) {
            return;
        }

        if ((int) $roles[$chatId]['trading']['topic_id'] !== $topicId) {
            return;
        }

        $author = $message['from'] ?? [];
        $authorId = isset($author['id']) ? (int) $author['id'] : 0;

        $name = '';
        if (isset($author['first_name']) && $author['first_name'] !== '') {
            $name = (string) $author['first_name'];
        } elseif (isset($author['username']) && $author['username'] !== '') {
            $name = '@' . (string) $author['username'];
        } elseif ($authorId !== 0) {
            $name = 'کاربر ' . $authorId;
        }

        $totalScore = 0;
        $ratings = $this->storage->load('ratings');
        if (is_array($ratings) && isset($ratings[$chatId]) && is_array($ratings[$chatId])) {
            foreach ($ratings[$chatId] as $entry) {
                if (!is_array($entry)) {
                    continue;
                }
                if ((int) ($entry['author_id'] ?? 0) !== $authorId) {
                    continue;
                }
                $stats = $entry['stats'] ?? [];
                $totalScore += (int) ($stats['sum'] ?? 0);
            }
        }

        if ($totalScore < 0) {
            $totalScore = 0;
        }

        if ($totalScore >= 1000) {
            $levelLabel = 'سطح ۵ - الگو 👑';
        } elseif ($totalScore >= 600) {
            $levelLabel = 'سطح ۴ - باتجربه 🧠';
        } elseif ($totalScore >= 300) {
            $levelLabel = 'سطح ۳ - فعال منظم 🔥';
        } elseif ($totalScore >= 100) {
            $levelLabel = 'سطح ۲ - در حال رشد 📈';
        } else {
            $levelLabel = 'سطح ۱ - تازه‌وارد 🔰';
        }

        $keyboard = [
            'inline_keyboard' => [
                [
                    ['text' => '⭐ 1', 'callback_data' => 'rate:' . $chatId . ':' . $topicId . ':' . $messageId . ':1'],
                    ['text' => '⭐ 2', 'callback_data' => 'rate:' . $chatId . ':' . $topicId . ':' . $messageId . ':2'],
                    ['text' => '⭐ 3', 'callback_data' => 'rate:' . $chatId . ':' . $topicId . ':' . $messageId . ':3'],
                    ['text' => '⭐ 4', 'callback_data' => 'rate:' . $chatId . ':' . $topicId . ':' . $messageId . ':4'],
                    ['text' => '⭐ 5', 'callback_data' => 'rate:' . $chatId . ':' . $topicId . ':' . $messageId . ':5'],
                ],
            ],
        ];
        $text = "⭐️ به این تحلیل چه امتیازی می‌دهید؟ (۱ تا ۵ ستاره)";

        $this->client->sendMessage($chatId, $text, [
            'reply_to_message_id' => $messageId,
            'reply_markup' => $keyboard,
        ]);
    }

    private function dispatchCommand(string $text, array $update): void
    {
        $parts = explode(' ', $text, 2);
        $command = strtolower($parts[0]);
        $command = explode('@', $command)[0];

        foreach ($this->modules as $module) {
            if ($module->supports($command)) {
                $module->handle($command, $update, $this);
                return;
            }
        }
    }

    public function registerUser(array $from): void
    {
        if (!isset($from['id'])) {
            return;
        }
        $userId = (int) $from['id'];
        $isMentor = $this->isAdmin($userId);

        $this->storage->update('users', function (array $users) use ($userId, $from, $isMentor): array {
            $existing = $users[$userId] ?? [];
            $now = (new \DateTimeImmutable())->format('Y-m-d H:i:s');
            $role = $isMentor ? 'mentor' : 'student';

            $users[$userId] = [
                'id' => $userId,
                'is_bot' => (bool) ($from['is_bot'] ?? false),
                'first_name' => (string) ($from['first_name'] ?? ''),
                'last_name' => (string) ($from['last_name'] ?? ''),
                'username' => (string) ($from['username'] ?? ''),
                'language_code' => (string) ($from['language_code'] ?? ''),
                'role' => $role,
                'is_mentor' => $isMentor,
                'created_at' => $existing['created_at'] ?? $now,
                'updated_at' => $now,
            ];

            return $users;
        });
    }

    private function isAdmin(int $userId): bool
    {
        $adminIds = $this->config['admin_ids'] ?? [];
        if (!is_array($adminIds)) {
            return false;
        }
        $adminIds = array_map('intval', $adminIds);
        return in_array($userId, $adminIds, true);
    }

    private function handleAdminKeywords(string $text, array $update): void
    {
        $message = $update['message'] ?? [];
        $from = $message['from'] ?? [];

        if (!isset($from['id'])) {
            return;
        }

        $userId = (int) $from['id'];
        if (!$this->isAdmin($userId)) {
            return;
        }

        $normalized = $text;

        if (strpos($normalized, 'پنل') !== false) {
            $this->dispatchCommand('/admin', $update);
            return;
        }

        if (strpos($normalized, 'نصب ربات') !== false) {
            $this->dispatchCommand('/install_group', $update);
            return;
        }
    }

    private function handleUserKeywords(string $text, array $update): void
    {
        $message = $update['message'] ?? [];
        $from = $message['from'] ?? [];

        if (!isset($from['id'])) {
            return;
        }

        $normalized = $text;

        if (strpos($normalized, 'امار من') !== false || strpos($normalized, 'آمار من') !== false) {
            $userId = (int) $from['id'];
            $chat = $message['chat'] ?? [];
            $chatId = isset($chat['id']) ? (int) $chat['id'] : 0;
            $chatType = isset($chat['type']) ? (string) $chat['type'] : '';

            $isAdmin = $this->isAdmin($userId);

            if ($isAdmin && ($chatType === 'group' || $chatType === 'supergroup')) {
                if ($chatId !== 0) {
                    $textReply = 'شما منتور تیم هستید.';
                    $this->client->sendMessage($chatId, $textReply);
                }
                return;
            }

            $updateCopy = $update;
            $updateCopy['message']['text'] = '/student';
            $this->dispatchCommand('/student', $updateCopy);
            return;
        }

        if (strpos($normalized, 'لیدربورد') !== false) {
            $updateCopy = $update;
            $updateCopy['message']['text'] = '/leaderboard';
            $this->dispatchCommand('/leaderboard', $updateCopy);
        }
    }

    private function updateInactivityOnActivity(int $chatId, int $userId): void
    {
        $settings = $this->storage->load('settings');
        if (!isset($settings['main_group']['id'])) {
            return;
        }

        $mainGroupId = (int) $settings['main_group']['id'];
        if ($mainGroupId !== $chatId) {
            return;
        }

        if ($this->isAdmin($userId)) {
            return;
        }

        $now = (new \DateTimeImmutable())->format('Y-m-d H:i:s');

        $this->storage->update('inactivity_status', function (array $data) use ($chatId, $userId, $now): array {
            if (!isset($data[$chatId])) {
                $data[$chatId] = [];
            }

            if (!isset($data[$chatId][$userId]) || !is_array($data[$chatId][$userId])) {
                $data[$chatId][$userId] = [
                    'user_id' => $userId,
                    'chat_id' => $chatId,
                    'last_activity_at' => $now,
                    'inactivity_days' => 0,
                    'pv_available' => null,
                    'warned_stage' => 0,
                    'last_warning_at' => '',
                    'admin_notified' => false,
                ];

                return $data;
            }

            $entry =& $data[$chatId][$userId];
            $entry['last_activity_at'] = $now;
            $entry['inactivity_days'] = 0;
            $entry['warned_stage'] = 0;
            $entry['admin_notified'] = false;

            return $data;
        });
    }

    private function logError(string $message): void
    {
        $dir = $this->config['data_dir'] ?? null;
        if (!$dir) {
            return;
        }
        $path = rtrim($dir, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . 'bot_errors.log';
        $line = '[' . date('Y-m-d H:i:s') . '] ' . $message . PHP_EOL;
        file_put_contents($path, $line, FILE_APPEND);
    }
}
